//
// (c) 2020 wesolutions GmbH
// All rights reserved
//

import QtQuick 2.12
import QtQuick.Layouts 1.12

import wesual.Create 1.0
import wesual.Stage  1.0
import wesual.Ui     1.0

import wesual.Ui.Private 1.0

import wesual.data.Core 1.0

UiVerticalScrollView {
    id : list

    property SizeDatabase sizeDatabase
    property SizeInfo activeSize

    readonly property alias preferredHeight : sizes.implicitHeight

    signal sizePicked(SizeInfo size)

    function createCustomSize(size) {
        __customSize = customSizeCreator.createObject(
                    sizes, { "size" : size });
    }

    property Item __customSize

    contentItem : Column {
        id : sizes

        spacing : -1

        Repeater {
            model : list.sizeDatabase.sizes
            delegate : sizeDelegate
        }

        PersistentTypeListing {
            id : customSizeList

            context : Creator.context
            type : "customsize"
        }

        DocumentSorter {
            id : sortedSizes

            delay : 0
            documents : customSizeList.documents

            SizeInfoSortCriterion {
                property : "sizeInfo"
            }
        }

        Item {
            height : 26
            width : sizes.width
            visible : list.__customSize !== null ||
                      customSizeList.documents.length > 0

            UiLabel {
                text : qsTrId("88b43a0a94497fb4")
                x : 4
                y : 9
            }
        }

        Repeater {
            model : sortedSizes.result
            delegate : customSizeDelegate
        }

        Component {
            id : sizeDelegate

            MouseArea {
                id : delegate

                readonly property bool active :
                    modelData === list.activeSize

                anchors {
                    left : parent ? parent.left : undefined
                    right : parent ? parent.right : undefined
                }
                hoverEnabled : true
                implicitWidth : 200
                implicitHeight : delegateContent.implicitHeight + 8

                onClicked : {
                    forceActiveFocus();
                    list.sizePicked(modelData);
                }

                UiBorder {
                    anchors.fill : delegate
                    style : UiBorder.Solid
                    borderWidth : 1
                    color : delegate.active
                            ? UiColors.getColor(UiColors.SelectionGreen)
                            : UiColors.getColor(UiColors.HoverGreen)
                    visible : delegate.active || delegate.containsMouse
                }

                RowLayout {
                    id : delegateContent

                    anchors.fill : parent
                    anchors.margins : 4
                    spacing : 4

                    SizeIcon {
                        iconSize : UiIcon.Icon_24x24
                        size : modelData

                        Layout.preferredWidth : iconSize
                        Layout.preferredHeight : iconSize
                        Layout.rightMargin : 1
                    }

                    UiText {
                        text : modelData.resolutionName.length > 0
                               ? modelData.resolutionName
                               : modelData.resolution
                    }

                    Rectangle {
                        color : "#AAA"

                        Layout.preferredHeight : 14
                        Layout.preferredWidth : 1
                        Layout.topMargin : 1
                    }

                    UiText {
                        text : modelData.orientationName + " " +
                               modelData.aspectRatioName

                        Layout.fillWidth : true
                    }
                }
            }
        }

        Component {
            id : customSizeDelegate

            MouseArea {
                id : delegate

                readonly property bool active :
                    modelData.sizeInfo === list.activeSize

                function activate() {
                    list.sizePicked(modelData.sizeInfo);
                }

                anchors {
                    left : parent ? parent.left : undefined
                    right : parent ? parent.right : undefined
                }
                hoverEnabled : true
                implicitWidth : 200
                implicitHeight : delegateContent.implicitHeight + 8

                onClicked : {
                    forceActiveFocus();
                    activate();
                }

                UiBorder {
                    anchors.fill : delegate
                    style : UiBorder.Solid
                    borderWidth : 1
                    color : delegate.active
                            ? UiColors.getColor(UiColors.SelectionGreen)
                            : UiColors.getColor(UiColors.HoverGreen)
                    visible : delegate.active ||
                              delegate.containsMouse ||
                              contextMenu_.menuVisible
                }

                RowLayout {
                    id : delegateContent

                    anchors.fill : parent
                    anchors.margins : 4
                    spacing : 4

                    SizeIcon {
                        iconSize : UiIcon.Icon_24x24
                        size : modelData.sizeInfo

                        Layout.preferredWidth : iconSize
                        Layout.preferredHeight : iconSize
                        Layout.rightMargin : 1
                    }

                    Loader {
                        id : sizeLoader

                        property point pos

                        sourceComponent : sizeComponent

                        onItemChanged : {
                            if (item.editAt) {
                                item.editAt(pos);
                            }
                        }
                    }

                    Rectangle {
                        color : "#AAA"

                        Layout.preferredHeight : 14
                        Layout.preferredWidth : 1
                        Layout.topMargin : 1
                    }

                    Loader {
                        id : labelLoader

                        sourceComponent : labelComponent

                        Layout.fillWidth : true
                    }

                    UiContextMenuTrigger {
                        contextMenu : contextMenu_
                        opacity : delegate.containsMouse ||
                                  contextMenu_.menuVisible ? 1 : 0
                    }

                    Component {
                        id : labelComponent

                        UiText {
                            text : modelData.label
                            elide : Text.ElideMiddle

                            MouseArea {
                                anchors.fill : parent

                                onPressed : forceActiveFocus()
                                onClicked : delegate.activate()
                                onDoubleClicked : labelLoader.sourceComponent =
                                                  labelEditorComponent
                            }
                        }
                    }

                    Component {
                        id : sizeComponent

                        UiText {
                            text : modelData.sizeInfo.resolution
                            elide : Text.ElideMiddle

                            MouseArea {
                                anchors.fill : parent
                                onPressed : forceActiveFocus()
                                onClicked : delegate.activate()
                                onDoubleClicked : {
                                    sizeLoader.pos = Qt.point(mouse.x, mouse.y);
                                    sizeLoader.sourceComponent =
                                                  sizeEditorComponent;
                                }
                            }
                        }
                    }

                    Component {
                        id : sizeEditorComponent

                        SizeEditor {
                            function done() {
                                sizeLoader.sourceComponent = sizeComponent;
                            }

                            size : modelData.sizeInfo
                            font : UiFonts.getFont(UiFonts.Light, 14)
                            onEditFinished : {
                                if (size) {
                                    Creator.context.beginRecordChanges();
                                    modelData.size = size.size;
                                    Creator.context.endRecordChanges();
                                }
                                done();
                            }
                        }
                    }

                    Component {
                        id : labelEditorComponent

                        TextInput {
                            id : labelEditor

                            function tryCommit() {
                                if (text.length > 0) {
                                    Creator.context.beginRecordChanges();
                                    modelData.label = text;
                                    Creator.context.endRecordChanges();
                                }
                                done();
                            }

                            function done() {
                                labelLoader.sourceComponent = labelComponent;
                            }

                            Shortcut {
                                sequence : "Esc"
                                onActivated : labelEditor.done();
                            }

                            clip : true
                            color : "#222"
                            font : UiFonts.getFont(UiFonts.Light, 14)
                            selectByMouse : true
                            selectionColor : UiColors.getColor(
                                                 UiColors.SelectionGreen)
                            text : modelData.label

                            onActiveFocusChanged : {
                                if (!activeFocus) {
                                    labelEditor.tryCommit();
                                }
                            }

                            Component.onCompleted : {
                                forceActiveFocus();
                                cursorPosition = 0;
                                selectAll();
                            }

                            Keys.onEnterPressed : labelEditor.tryCommit()
                            Keys.onReturnPressed : labelEditor.tryCommit()
                        }
                    }

                    UiContextMenu {
                        id : contextMenu_

                        UiMenuItem {
                            iconName : "rename"
                            text : qsTrId("4893f9df64cdc64e")

                            onTriggered : {
                                labelLoader.sourceComponent =
                                        labelEditorComponent;
                            }
                        }

                        UiMenuItem {
                            iconName : "size"
                            text : qsTrId("4a9aa59fab431d43")

                            onTriggered : {
                                sizeLoader.sourceComponent =
                                        sizeEditorComponent;
                            }
                        }

                        UiMenuItem {
                            iconName : "delete"
                            text : qsTrId("067ce595b65acd59")

                            onTriggered: {
                                Creator.context.beginRecordChanges();
                                modelData.deleted = true;
                                Creator.context.endRecordChanges();
                            }
                        }
                    }
                }
            }
        }

        Component {
            id : customSizeCreator

            Item {
                id : customSize

                property SizeInfo size

                function tryCommit() {
                    if (label.text.length > 0) {
                        const ctx = Creator.context;

                        ctx.beginRecordChanges();
                        try {
                            const cs = ctx.createDocument("customsize");
                            cs.size = size.size;
                            cs.label = label.text;
                        } catch (err) {
                            console.error(err);
                        }
                        ctx.endRecordChanges();
                    }

                    customSize.destroy();
                }

                anchors {
                    left : parent ? parent.left : undefined
                    right : parent ? parent.right : undefined
                }
                implicitWidth : 200
                implicitHeight : delegateContent.implicitHeight + 8

                Shortcut {
                    sequence : "Esc"
                    onActivated : customSize.destroy()
                }

                Timer {
                    running : true
                    repeat : false
                    interval : 150
                    onTriggered : list.scrollTo(customSize)
                }

                RowLayout {
                    id : delegateContent

                    anchors.fill : parent
                    anchors.margins : 4
                    spacing : 4

                    SizeIcon {
                        iconSize : UiIcon.Icon_24x24
                        size : customSize.size

                        Layout.preferredWidth : iconSize
                        Layout.preferredHeight : iconSize
                        Layout.rightMargin : 1
                    }

                    UiText {
                        text : customSize.size.resolution
                    }

                    Rectangle {
                        color : "#AAA"

                        Layout.preferredHeight : 14
                        Layout.preferredWidth : 1
                        Layout.topMargin : 1
                    }

                    TextInput {
                        id : label

                        clip : true
                        color : "#222"
                        font : UiFonts.getFont(UiFonts.Light, 14)
                        selectByMouse : true
                        selectionColor : UiColors.getColor(
                                             UiColors.SelectionGreen)
                        text : qsTrId("b3661e7daa19922f")

                        onActiveFocusChanged : {
                            if (!activeFocus) {
                                customSize.tryCommit();
                            }
                        }

                        Component.onCompleted : {
                            forceActiveFocus();
                            cursorPosition = 0;
                            selectAll();
                        }

                        Layout.fillWidth : true

                        Keys.onEnterPressed : customSize.tryCommit()
                        Keys.onReturnPressed : customSize.tryCommit()
                    }
                }
            }
        }
    }
}
